﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Newtonsoft.Json;
using Sitecore;
using Sitecore.Data;
using Sitecore.Data.Managers;
using Smartling.Connector.Diagnostics;
using Smartling.Connector.Extensions;
using Smartling.Connector.Model;
using Smartling.Connector.Security;
using Smartling.Connector.Translation;
using Smartling.Connector.Translation.Profile;

namespace Smartling.Connector.sitecore_modules.Shell.Smartling.Connector
{
  /// <summary>
  /// Summary description for Audit
  /// </summary>
  public class AuditApi : IHttpHandler
  {
    private const string JsonContentType = "application/json";
    private const string LengthParamName = "length";
    private const string StartParamName = "start";
    private const string DrawParamName = "draw";
    private const string ShortDateFormat = "dd MMM, yyyy";
    private const string LongDateFormat = "dd MMM, yyyy HH:mm:ss tt";
    private const string DateFormatString = "<span style=\"display:none\">{0}</span><span title='{1}'>{2}</span>";
    private const string ExternalLinkTemplate = "<a href='{0}' target='_blank'>{1}</a>";
    private readonly Dictionary<string, string> languages = new Dictionary<string, string>();

    public void ProcessRequest(HttpContext context)
    {
      if (!SecurityManager.AccessAllowed())
      {
        return;
      }

      try
      {
        Initialize();
        var query = HttpContext.Current.Request.QueryString;
        var request = new AuditLogPageRequest();
        request.DisplayLength = MainUtil.GetInt(query[LengthParamName], 0);
        request.DisplayStart = MainUtil.GetInt(query[StartParamName], 0);
        request.Search = GetSearch();

        var profile = new ProfileService().GetAllProfiles().First(x => x.Name == request.Search.Profile);

        var page = profile.GetAuditClient()
          .GetPage(BuildSearchQuery(request, profile), "actionTime:desc", request.DisplayLength, request.DisplayStart);
        var items = page.items;

        var data = GetData(items.ToList(), profile);
        data.draw = MainUtil.GetInt(query[DrawParamName], 0);
        data.recordsTotal = page.totalCount;
        data.recordsFiltered = page.totalCount;

        var result = JsonConvert.SerializeObject((object) data);
        context.Response.ContentType = JsonContentType;
        context.Response.Write(result);
      }
      catch (Exception ex)
      {
        SmartlingLogger.Log.Error("Failed to load Audit Logs", ex);
        var result = JsonConvert.SerializeObject((object)new List<AuditLog>());
        context.Response.ContentType = JsonContentType;
        context.Response.Write(result);
      }
    }

    private static Dictionary<string, string> BuildSearchQuery(AuditLogPageRequest request, TranslationProfile profile)
    {
      Dictionary<string, string> searchQuery = new Dictionary<string, string>();
      searchQuery.Add("envId", profile.BucketName);

      if (!string.IsNullOrEmpty(request.Search.Query))
      {
        searchQuery.Add(string.Empty, $"{request.Search.Query}*");
      }

      if (!string.IsNullOrEmpty(request.Search.Item))
      {
        searchQuery.Add("clientData.ItemId|clientData.Path|fileUri", $"{request.Search.Item}*");
      }

      if (!string.IsNullOrEmpty(request.Search.Submitter))
      {
        searchQuery.Add("clientUserId", $"{request.Search.Submitter}*");
      }

      if (!string.IsNullOrEmpty(request.Search.Job))
      {
        searchQuery.Add("translationJobName|translationJobUid", $"{request.Search.Job}*");
      }

      if (request.Search.Language != null && request.Search.Language.Any())
      {
        searchQuery.Add("targetLocaleIds", string.Join("|", request.Search.Language.ToArray()));
      }

      if (request.Search.EventType != null && request.Search.EventType.Any())
      {
        searchQuery.Add("actionType", string.Join("|", request.Search.EventType.ToArray()));
      }

      return searchQuery;
    }

    private void Initialize()
    {
      foreach (var language in LanguageManager.GetLanguages(Database.GetDatabase(ModuleConstants.DefaultDatabase)))
      {
        languages.Add(language.Name, language.CultureInfo.DisplayName);
      }
    }

    private static AuditLogSearch GetSearch()
    {
      var filters = new AuditLogSearch();
      var query = HttpContext.Current.Request.QueryString;

      filters.Query = query["query"];
      filters.Item = query["item"];
      filters.Job = query["job"];
      filters.Submitter = query["submitter"];
      filters.Profile = query["profile"];

      if (!string.IsNullOrEmpty(query["language[]"]))
      {
        filters.Language = query["language[]"].Split(',').ToList();
      }

      if (!string.IsNullOrEmpty(query["eventType[]"]))
      {
        filters.EventType = query["eventType[]"].Split(',').ToList();
      }

      return filters;
    }

    private RootObject GetData(List<AuditLog> items, TranslationProfile profile)
    {
      var dataObject = new RootObject();
      dataObject.data = new List<List<string>>();
      var accountUid = profile.GetProjectClient().GetProjectData().accountUid;

      foreach (var item in items)
      {
        var list = new List<string>();
        list.Add(string.Format(ExternalLinkTemplate, item.clientData.ItemId.ContentEditorUrl(item.sourceLocaleId), item.clientData.Path));
        list.Add(GetLanguage(item.sourceLocaleId));
        list.Add(GetLanguage(item.targetLocaleIds?.FirstOrDefault()));
        list.Add(string.Format(ExternalLinkTemplate, item.clientData.ItemId.ContentEditorUrl(item.sourceLocaleId, item.clientData.SourceVersion.GetValueOrDefault()), item.clientData.SourceVersion));
        list.Add(string.Format(ExternalLinkTemplate, item.clientData.ItemId.ContentEditorUrl(item.sourceLocaleId, item.clientData.TargetVersion.GetValueOrDefault()), item.clientData.TargetVersion));
        list.Add(string.Format(ExternalLinkTemplate, item.translationJobUid.JobUrl(profile.ProjectId, accountUid), item.translationJobName));
        list.Add(item.clientUserId);
        list.Add(item.actionType.CapitalizeFirstLetter().ReplaceUnderscoreWithSpace());
        list.Add(string.Format(DateFormatString, DateUtil.ToIsoDate(item.actionTime), item.actionTime.ToString(LongDateFormat), item.actionTime.ToString(ShortDateFormat)));
        list.Add(item.description);
   
        dataObject.data.Add(list);
      }

      return dataObject;
    }

    private string GetLanguage(string language)
    {
      if (string.IsNullOrEmpty(language))
      {
        return language;
      }

      if (languages.ContainsKey(language))
      {
        return languages[language];
      }

      return language;
    }
    
    public bool IsReusable
    {
      get
      {
        return false;
      }
    }
  }

  public class RootObject
  {
    public int draw;
    public long recordsTotal;
    public long recordsFiltered;
    public List<List<string>> data { get; set; }
  }
}